#include "stdafx.h"
#include "version.h"

#include "rwexp.h"
#include "utilities.h"
#include "material.h"
#include "matercst.h"
#include "rpmatfx.h"

static RwBool
IsStandardMaterial(Mtl *maxMaterial)
{
    if(!maxMaterial)
    {
        return(FALSE);
    }

    /* Standard material */
    if (maxMaterial->ClassID() == Class_ID(DMTL_CLASS_ID, 0) ||
        maxMaterial->IsSubClassOf(Class_ID(DMTL_CLASS_ID, 0)))
    {
        return(TRUE);
    }

    return(FALSE);
}

void
MaterialGetColour(Mtl *mtlpMaterial, RwRGBA *cpColour, RwBool bOpacityOnly)
{
    RwRGBAReal nColour;
    RwReal     rOpacity = 1.0f;
    Color      maxColour = Color(1.0f, 1.0f, 1.0f);

    /* get max opacity & colour*/
    if(IsStandardMaterial(mtlpMaterial))
    {
        rOpacity  = (RwReal)(((StdMat *)mtlpMaterial)->GetOpacity(0));
        if(!bOpacityOnly)
        {
            maxColour = ((StdMat *)mtlpMaterial)->GetDiffuse(0);
        }
    }

    nColour.red   = (RwReal)(maxColour.r);
    nColour.green = (RwReal)(maxColour.g);
    nColour.blue  = (RwReal)(maxColour.b);
    nColour.alpha = rOpacity;

    RwRGBAFromRwRGBAReal(cpColour, &nColour);
}

RwBool
GetRwSurfaceProperties(Mtl *maxMaterial, RwSurfaceProperties *spProperties)
{
    /* Sanity check. */
    if(!maxMaterial || !spProperties)
    {
        return(FALSE);
    }

    /* Assign the surface properties. */
    spProperties->ambient = 1.0f;
    spProperties->diffuse = 1.0f;

    if (maxMaterial->ClassID() == Class_ID(DMTL_CLASS_ID, 0))
    {
        spProperties->specular = ((StdMat *)maxMaterial)->GetShinStr(0);
    }
    else
    {
        spProperties->specular = 1.0f;
    }

    return(TRUE);
}

/* 
    Get the sub material returns NULL in case of errors such
    as nested sub materials 
*/
Mtl *
GetSubMaterial(Mtl *mat, MtlID mtId)
{
    Mtl *baseMaterial = NULL;

    if (mat)
    {
        Class_ID clID = mat->ClassID();

        /* RPLT - Also allow the TS2 Material */
        if (mat->ClassID() == Class_ID(DMTL_CLASS_ID, 0))
        {
            baseMaterial = mat;
        }
        else if (mat->ClassID() == Class_ID(MULTI_CLASS_ID, 0))
        {            
            Mtl *subMtl = NULL;

            if (mat->NumSubMtls())
            {
                if (mtId >= mat->NumSubMtls()) mtId %= mat->NumSubMtls();
            }
            else
            {
                baseMaterial = NULL;
                return baseMaterial;
            }
            subMtl = mat->GetSubMtl(mtId);
            if (subMtl->ClassID() == Class_ID(MULTI_CLASS_ID, 0))
            {
                char temp[256];
                sprintf(temp,"RenderWare %s Bsp Exporter",VersionNumberText);
                MessageBox(NULL, "Nested Multi/Sub-Object Material, Using a default",
                    _T(temp), MB_OK );
                baseMaterial = NULL;
            }
            else
            {
                baseMaterial = subMtl;
            }
        }
    }

    return (baseMaterial);
}

Texmap *
GetStdMatTexmap(Mtl *mtl, int texMapIndex)
{
    if (!mtl)
    {
        return NULL;
    }

    if (mtl->ClassID() == Class_ID(DMTL_CLASS_ID, 0))
    {
        if (((StdMat *)mtl)->MapEnabled(texMapIndex))
        {
            return ((StdMat *)mtl)->GetSubTexmap(texMapIndex);
        }
        else
        {
            return NULL;
        }
    }
    else
    {
        return NULL;
    }
}

/* This function returns whether the texmap class passed in is
   supported. This is used to determine whether to set the 
   material color or just opacity */
RwBool
SupportedTexmapClass(Texmap *tex)
{
    if (!tex)
    {
        return FALSE;
    }

    if (tex->ClassID() == Class_ID(BMTEX_CLASS_ID, 0))
    {
        return TRUE;
    }

    return FALSE;
}

BOOL 
DoubleSidedMaterial(Mtl *mat, MtlID matID)
{
    Mtl *submtl = GetSubMaterial(mat, matID);
    if (!submtl) 
    {
        return FALSE;
    }

    if (submtl->ClassID() == Class_ID(DMTL_CLASS_ID, 0))
    {
        return ((StdMat *)submtl)->GetTwoSided();
    }
    else
    {
        return FALSE;
    }
}

//Get the factor of a material texture map, returns 1.0 for unknown material types
RwReal
GetMtlTexmapAmt(Mtl *mat, int mapType, int time)
{
    if (mat->ClassID() == Class_ID(DMTL_CLASS_ID, 0))
    {
        return ((StdMat *)mat)->GetTexmapAmt(mapType,time);
    }
    else
    {
        return 1.0f;
    }
}

/* Calculate a self illumination value over the entire mesh
   if different materials have different values we won't self
   illuminate */
float
getMeshSelfIllumination(Mesh *mesh, Mtl *mat)
{
    int face;
    float selfIllum = -1.0f;

    for (face = 0; face < mesh->getNumFaces(); face++)
    {
        int matID = mesh->getFaceMtlIndex(face);
        Mtl *submtl = GetSubMaterial(mat, matID);

        if ( submtl && (submtl->ClassID() == Class_ID(DMTL_CLASS_ID, 0)) )
        {
            if (selfIllum == -1.0f)
            {
                selfIllum = ((StdMat *)submtl)->GetSelfIllum(0);
            }
            else
            {
                if (selfIllum != ((StdMat *)submtl)->GetSelfIllum(0))
                {
                    return 0.0f;
                }
            }
        }
        else
        {
            return 0.0f;
        }
    }
    return selfIllum;
}

/* code for checking if materials with color exist */
RpMaterial *
checkMaterialColors(RpMaterial *material, void *pData)
{
    RwBool *colored = (RwBool *)pData;
    const RwRGBA *color;

    color = RpMaterialGetColor(material);
    if (color->red != 255 ||
        color->green != 255 ||
        color->blue != 255 ||
        color->alpha != 255)
    {
        *colored = TRUE;
        return NULL;
    }
    
    return material;
}


/* code for checking if materials with effects exist */
RpMaterial *
checkMaterialEffects(RpMaterial *material, void *pData)
{
    RwBool *effects = (RwBool *)pData;    

    if (RpMatFXMaterialGetEffects(material) != rpMATFXEFFECTNULL)
    {
        *effects = TRUE;
        return NULL;
    }

    return material;
}